<?php

/*
 * This file is part of SwiftMailer.
 * (c) 2004-2009 Chris Corbyn
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

/**
 * NewsletterPro_Swift Mailer class.
 *
 * @author  Chris Corbyn
 */
class NewsletterPro_Swift_Mailer
{
    /** The Transport used to send messages */
    private $_transport;

    /**
     * Create a new Mailer using $transport for delivery.
     *
     * @param NewsletterPro_Swift_Transport $transport
     */
    public function __construct(NewsletterPro_Swift_Transport $transport)
    {
        $this->_transport = $transport;
    }

    /**
     * Create a new Mailer instance.
     *
     * @param NewsletterPro_Swift_Transport $transport
     *
     * @return NewsletterPro_Swift_Mailer
     */
    public static function newInstance(NewsletterPro_Swift_Transport $transport)
    {
        return new self($transport);
    }

    /**
     * Create a new class instance of one of the message services.
     *
     * For example 'mimepart' would create a 'message.mimepart' instance
     *
     * @param string $service
     *
     * @return object
     */
    public function createMessage($service = 'message')
    {
        return NewsletterPro_Swift_DependencyContainer::getInstance()
            ->lookup('message.'.$service);
    }

    /**
     * Send the given Message like it would be sent in a mail client.
     *
     * All recipients (with the exception of Bcc) will be able to see the other
     * recipients this message was sent to.
     *
     * Recipient/sender data will be retrieved from the Message object.
     *
     * The return value is the number of recipients who were accepted for
     * delivery.
     *
     * @param NewsletterPro_Swift_Mime_Message $message
     * @param array              $failedRecipients An array of failures by-reference
     *
     * @return int
     */
    public function send(NewsletterPro_Swift_Mime_Message $message, &$failedRecipients = null)
    {
        $failedRecipients = (array) $failedRecipients;

        if (!$this->_transport->isStarted()) {
            $this->_transport->start();
        }

        $sent = 0;

        try {
            $sent = $this->_transport->send($message, $failedRecipients);
        } catch (NewsletterPro_Swift_RfcComplianceException $e) {
            foreach ($message->getTo() as $address => $name) {
                $failedRecipients[] = $address;
            }
        }

        return $sent;
    }

    /**
     * Register a plugin using a known unique key (e.g. myPlugin).
     *
     * @param NewsletterPro_Swift_Events_EventListener $plugin
     */
    public function registerPlugin(NewsletterPro_Swift_Events_EventListener $plugin)
    {
        $this->_transport->registerPlugin($plugin);
    }

    /**
     * The Transport used to send messages.
     *
     * @return NewsletterPro_Swift_Transport
     */
    public function getTransport()
    {
        return $this->_transport;
    }
}
